<?php
/**
 * Class WSAL_LogsManagement
 *
 * @since   4.3.3
 * @package wsal
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Extension: Logs Management
 *
 * Log management extension for wsal.
 *
 * @since 4.3.3
 */
class WSAL_LogsManagement {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_filter( 'wsal_setting_tabs', array( $this, 'add_logs_management_tab' ), 10, 1 );
		add_filter( 'wp_ajax_wsal_delete_data_from_logs', array( $this, 'delete_data_from_logs' ), 10, 1 );

		// Add scripts.
		add_action( 'admin_init', array( $this, 'setup_logs_management' ) );
	}

	/**
	 * Checks if the extension is active.
	 *
	 * @return bool True if the extension is active
	 */
	public function is_active() {
		return wsal_freemius()->is_plan_or_trial__premium_only( 'professional' );
	}

	/**
	 * Add scripts and styles for this extension.
	 *
	 * @return void
	 */
	public function setup_logs_management() {

		// Get current tab.
		$current_tab = ( isset( $_GET['tab'] ) ) ? \sanitize_text_field( \wp_unslash( $_GET['tab'] ) ) : '';

		if ( 'logs-management' !== $current_tab ) {
			return;
		}

		wp_enqueue_script(
			'logs-management',
			WSAL_BASE_URL . '/extensions/logs-management/js/logs-management.js',
			array(),
			WSAL_VERSION,
			true
		);

		// Passing nonce for security to JS file.
		$wsal_data = array(
			'wp_nonce' => wp_create_nonce( 'wsal-exclude-nonce' ),
		);
		wp_localize_script( 'logs-management', 'wsal_data', $wsal_data );
		wp_enqueue_style( 'logs-management-style', WSAL_BASE_URL . '/extensions/logs-management/css/style.css', array(), WSAL_VERSION );
	}

	/**
	 * Add log management tab to WSAL settings.
	 *
	 * @param array $wsal_setting_tabs WSAL settings tab.
	 *
	 * @return array - Tabs, plus our tab.
	 */
	public function add_logs_management_tab( $wsal_setting_tabs ) {
		$wsal_setting_tabs['logs-management'] = array(
			'name'     => esc_html__( 'Delete activity log data', 'wp-security-audit-log' ),
			'link'     => add_query_arg( 'tab', 'logs-management' ),
			'render'   => array( $this, 'logs_management_tab' ),
			'save'     => false,
			'priority' => 70,
		);
		return $wsal_setting_tabs;
	}

	/**
	 * Handle content.
	 *
	 * @return void
	 */
	public function logs_management_tab() {
		$this->tab_content();
	}

	/**
	 * The actual settings/tab content.
	 *
	 * @return void
	 */
	private function tab_content() {

		$disabled  = ! $this->is_active() ? 'disabled' : '';
		$admin_url = ! is_multisite() ? 'admin_url' : 'network_admin_url';
		$buy_now   = add_query_arg( 'page', 'wsal-auditlog-pricing', $admin_url( 'admin.php' ) );
		$html_tags = WpSecurityAuditLog::get_instance()->allowed_html_tags;
		$nonce     = wp_create_nonce( 'wsal-delete-log-data' );

		$tab_info_msg = esc_html__( 'Use the below to delete data from the activity log. Note that once the data is deleted it cannot be recovered unless you have a backup.', 'wp-security-audit-log' );
		if ( $disabled ) {
			/* Translators: Upgrade now hyperlink. */
			$tab_info_msg = sprintf( esc_html__( 'Logs management is available in the Professional and Business Plans. %s to configure and receive this feature.', 'wp-security-audit-log' ), '<a href="' . $buy_now . '">' . esc_html__( 'Upgrade now', 'wp-security-audit-log' ) . '</a>' );
		}
		?>

		<?php
		// Show if applicable.
		if ( ! $disabled ) :
			?>
		<p class="description"> <?php echo wp_kses( $tab_info_msg, $html_tags ); ?></p>
		<table class="form-table wsal-tab logs-management-settings">
		<tr>
				<th><label><?php esc_html_e( 'Delete activity log data from a specific user', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="ExUserQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter username', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-user-log-data" class="button-primary" value="Delete user data" data-delete-log-data-input="#ExUserQueryBox" data-delete-log-data-type="user" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the user\'s username above and click delete data to delete all the events in the activity log that either have been generated by this user.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete activity log data about a specific user', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="ExUserSubjectQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter username', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-user-log-data" class="button-primary" value="Delete user data" data-delete-log-data-input="#ExUserSubjectQueryBox" data-delete-log-data-type="user-subject" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the user\'s username above and click delete data to delete all the events in the activity log that this user is referenced in.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete activity log data from a specific IP address', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="IpAddrQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter IP', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-ip-log-data" class="button-primary" value="Delete IP address data" data-delete-log-data-input="#IpAddrQueryBox" data-delete-log-data-type="ip" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the IP address above and click delete data to delete all the events in the activity log that have been generated from this IP address.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete activity log data about a specific IP address', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="IpAddrSubjectQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter IP', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-ip-log-data" class="button-primary" value="Delete IP address data" data-delete-log-data-input="#IpAddrSubjectQueryBox" data-delete-log-data-type="ip-subject" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the IP address above and click delete data to delete all the events in the activity log that have this IP address is referenced in it.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete events with a specific severity level', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="SeveritiesQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter severity', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-severity-data" class="button-primary" value="Delete Severity data" data-delete-log-data-input="#SeveritiesQueryBox" data-delete-log-data-type="severity" data-nonce="<?php echo esc_attr( $nonce ); ?>">							
						<p class="description">
							<?php esc_html_e( 'Specify the severity level above and click delete data to delete all the events in the activity log of that severity level.', 'wp-security-audit-log' ); ?>
						</p>				
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete events with a specific event type', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="EventTypeQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter event type', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-event-type-data" class="button-primary" value="Delete event type data" data-delete-log-data-input="#EventTypeQueryBox" data-delete-log-data-type="event_type" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the event type above and click delete data to delete all the events in the activity log of the specified event type.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete events about a specific object', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="ObjectTypeQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter object', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-object-data" class="button-primary" value="Delete object data" data-delete-log-data-input="#ObjectTypeQueryBox" data-delete-log-data-type="object" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the object above and click delete data to delete all the events in the activity log with the specific object.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>

			<tr>
				<th><label><?php esc_html_e( 'Delete events with a specific event ID', 'wp-security-audit-log' ); ?></label></th>
				<td>
					<fieldset <?php echo esc_attr( $disabled ); ?>>
						<input type="text" id="EventIDQueryBox" style="width: 250px;" placeholder="<?php esc_attr_e( 'Enter event ID', 'wp-security-audit-log' ); ?>"> <input type="button" id="delete-event-id-data" class="button-primary" value="Delete event IDs data" data-delete-log-data-input="#EventIDQueryBox" data-delete-log-data-type="event_id" data-nonce="<?php echo esc_attr( $nonce ); ?>">
						<p class="description">
							<?php esc_html_e( 'Specify the event ID above and click delete data to delete all the events in the activity log of that type.', 'wp-security-audit-log' ); ?>
						</p>
					</fieldset>
				</td>
			</tr>
		</table>
			<?php
		endif;
	}

	/**
	 * Simple function to remove log entries based on a specified criteria.
	 */
	public function delete_data_from_logs() {
		$wsal = WpSecurityAuditLog::get_instance();

		$nonce = ( isset( $_POST['nonce'] ) ) ? \sanitize_text_field( \wp_unslash( $_POST['nonce'] ) ) : '';

		// Check nonce.
		if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'wsal-delete-log-data' ) ) {
			wp_send_json_error( esc_html__( 'Nonce Verification Failed.', 'wp-security-audit-log' ) );
		}

		// Grab POSTed data.
		$type           = ( isset( $_POST['type'] ) ) ? \sanitize_text_field( \wp_unslash( $_POST['type'] ) ) : '';
		$item_to_remove = ( isset( $_POST['item_to_remove'] ) ) ? \sanitize_text_field( \wp_unslash( $_POST['item_to_remove'] ) ) : '';

		// We might populate this later.
		$delete_lines   = array();
		$prepared_query = '';

		// Gather correct table names.
		$occ_table  = $wsal->get_connector()->get_adapter( 'Occurrence' )->get_table();
		$meta_table = $wsal->get_connector()->get_adapter( 'Meta' )->get_table();

		$db_config = WSAL_Connector_ConnectorFactory::get_config(); // Get DB connector configuration.
		$connector = $wsal->get_connector( $db_config ); // Get connector for DB.
		$wsal_db   = $connector->get_connection(); 

		// Types we dont need special treatment for.
		$names = array(
			'ip'         => 'ClientIP',
			'severity'   => 'Severity',
			'object'     => 'Object',
			'event_type' => 'EventType',
		);

		// Build query based on selected type.
		if ( 'user' === $type ) {
			$user = get_user_by( 'login', $item_to_remove );
			// Bail if we still dont have a user by this point.
			if ( ! $user instanceof WP_User ) {
				$message = esc_html__( 'User not found.', 'wp-security-audit-log' );
				wp_send_json_error( $message );
			}
			$prepared_query = $wsal_db->prepare(
				"DELETE FROM $occ_table WHERE $occ_table.user_id = %d", // phpcs:ignore
				$user->ID
			);
		} elseif ( 'event_id' === $type ) {
			$prepared_query = $wsal_db->prepare(
				"DELETE FROM $occ_table WHERE $occ_table.alert_id = %s", // phpcs:ignore
				$item_to_remove
			);
		} elseif ( isset( $names[ $type ] ) ) {
			if ( 'object' === $type ) {
				$objects        = array_flip( $wsal->alerts->get_event_objects_data() );
				$item_to_remove = $objects[ $item_to_remove ];
				$prepared_query = $wsal_db->prepare(
					"DELETE FROM $occ_table WHERE $occ_table.object = %s", // phpcs:ignore
					$item_to_remove
				);
			} elseif ( 'event_type' === $type ) {
				$objects        = array_flip( $wsal->alerts->get_event_type_data() );
				$item_to_remove = $objects[ $item_to_remove ];
				$prepared_query = $wsal_db->prepare(
					"DELETE FROM $occ_table WHERE $occ_table.event_type = %s", // phpcs:ignore
					$item_to_remove
				);
			} elseif ( 'severity' === $type ) {
				$severity_obj = $wsal->constants->get_constant_by( 'name', 'WSAL_' . strtoupper( $item_to_remove ) );
				if ( ! is_null( $severity_obj ) ) {
					$prepared_query = $wsal_db->prepare(
						"DELETE FROM $occ_table WHERE $occ_table.severity = %d", // phpcs:ignore
						$severity_obj->value
					);
				}
			} else {
				$prepared_query = $wsal_db->prepare(
					"DELETE FROM $occ_table WHERE $occ_table.client_ip = %s",  // phpcs:ignore
					$item_to_remove
				);
			}
		} elseif ( 'user-subject' === $type ) {
			$concat_string  = '"' . $item_to_remove . '"';
			$prepared_query = "DELETE FROM $occ_table WHERE $occ_table.id IN ( SELECT occurrence_id FROM $meta_table meta WHERE ( meta.name = 'login' AND meta.value = '$item_to_remove' ) OR ( meta.name = 'NewUserData' AND LOCATE( $concat_string, meta.value ) ) OR ( meta.name = 'TargetUsername' AND meta.value = '$item_to_remove' ) )";
		} elseif ( 'ip-subject' === $type ) {
            $prepared_query  = "DELETE FROM $occ_table WHERE $occ_table.id IN ( SELECT occurrence_id FROM $meta_table meta WHERE ( meta.name = 'ClientIP' AND meta.value = '$item_to_remove' ) OR ( meta.name = 'IPAddress' AND meta.value = '$item_to_remove' ) OR ( meta.name = 'ip' AND meta.value = '$item_to_remove' ) )";
		}

		// Run the query.
		$delete_lines = ( ! empty( $prepared_query ) ) ? $wsal_db->query( $prepared_query ) : array(); // phpcs:ignore

		// Report back.
		if ( ! empty( $delete_lines ) ) {
			$message = ucwords( str_replace( '_', ' ', $type ) ) . ' ' . esc_html__( 'data deleted.', 'wp-security-audit-log' ) . ' ' . $delete_lines . esc_html__( ' rows removed', 'wp-security-audit-log' );

			// Trigger alert.
			$alert_data = array();

			if ( 'user' === $type ) {
				$user_data               = get_userdata( $user->ID );
				$alert_data['user']      = $item_to_remove;
				$alert_data['Role']      = is_array( $user_data->roles ) ? implode( ', ', $user_data->roles ) : $user_data->roles;
				$alert_data['FirstName'] = ( empty( $user_data->first_name ) ) ? esc_html__( 'Not provided', 'wp-security-audit-log' ) : $user_data->first_name;
				$alert_data['LastName']  = ( empty( $user_data->last_name ) ) ? esc_html__( 'Not provided', 'wp-security-audit-log' ) : $user_data->last_name;
				$wsal->alerts->trigger_event( 6038, $alert_data );
			} else {
				// Little tidy up for show.
				$alert_data['deleted_data_type'] = ucwords( str_replace( 'Ip', 'IP', str_replace( '_', ' ', $type ) ) );
				$alert_data['deleted_data']      = $item_to_remove;
				$wsal->alerts->trigger_event( 6039, $alert_data );
			}

			// Delete orphaned data.
			$wsal_db->query( "DELETE meta FROM $meta_table meta LEFT JOIN $occ_table occ ON occ.id = meta.occurrence_id WHERE occ.id IS NULL" ); // phpcs:ignore

			wp_send_json_success( $message );
		} else {
			$message = esc_html__( 'No events found.', 'wp-security-audit-log' );
			wp_send_json_error( $message );
		}
	}
}
